// backend/controllers/planController.js

const Plan = require("../models/planModel");

/**
 * @desc    Create a new subscription plan
 * @route   POST /api/plans
 * @access  Private (Super Admin)
 */
exports.createPlan = async (req, res) => {
  try {
    const {
      name,
      price,
      billing_interval,
      interval_count,
      student_limit,
      trial_days,
      features,
      status,
    } = req.body;

    if (
      !name ||
      price === undefined ||
      !billing_interval ||
      student_limit === undefined
    ) {
      return res.status(400).json({
        message:
          "Name, price, billing interval, and student limit are required.",
      });
    }

    const newPlan = await Plan.create({
      name,
      price,
      billing_interval,
      interval_count: interval_count || 1,
      student_limit,
      trial_days: trial_days || 0,
      features,
      status: status || "active",
    });

    res
      .status(201)
      .json({ message: "Plan created successfully.", plan: newPlan });
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error creating plan.", error: error.message });
  }
};

exports.getPlan = async (req, res) => {
  try {
    const plan = await Plan.findById(req.params.id);
    if (plan) {
      res.status(200).json(plan);
    } else {
      res.status(404).json({ message: "Plan not found." });
    }
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error fetching plan.", error: error.message });
  }
};

/**
 * @desc    Get all subscription plans
 * @route   GET /api/plans
 * @access  Private (Super Admin)
 */
exports.getAllPlans = async (req, res) => {
  try {
    const plans = await Plan.findAll();
    res.status(200).json(plans);
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error fetching plans.", error: error.message });
  }
};

/**
 * @desc    Update a subscription plan
 * @route   PUT /api/plans/:id
 * @access  Private (Super Admin)
 */
exports.updatePlan = async (req, res) => {
  try {
    const success = await Plan.update(req.params.id, req.body);
    if (success) {
      res.status(200).json({ message: "Plan updated successfully." });
    } else {
      res.status(404).json({ message: "Plan not found or no changes made." });
    }
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error updating plan.", error: error.message });
  }
};

/**
 * @desc    Delete a subscription plan
 * @route   DELETE /api/plans/:id
 * @access  Private (Super Admin)
 */
exports.deletePlan = async (req, res) => {
  try {
    // Add logic here to check if any active subscription is using this plan before deleting.
    const success = await Plan.delete(req.params.id);
    if (success) {
      res.status(200).json({ message: "Plan deleted successfully." });
    } else {
      res.status(404).json({ message: "Plan not found." });
    }
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error deleting plan.", error: error.message });
  }
};
